/*******************************************************************************
  FINDDATA - Program that searches recursively through directories
             searching for specified data

  (c) Copyright 2003 Theo Berkau(cwx@softhome.net)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*******************************************************************************/

#include <windows.h>
#include <stdio.h>
#include "util.h"

//////////////////////////////////////////////////////////////////////////////

unsigned char *work_buffer;
unsigned char *search_value;
unsigned long search_size=0;
FILE *fp;

void SearchPath(char *path);
void SearchDir(char *dir);
void SearchFile(char *filename, unsigned long size);
HANDLE GrabSubDirectory(char *path, HANDLE hFind, char *subdir);
void TruncateLastDir(char *buffer);

char outpath[MAXPATH];
char indrive[MAXDRIVE];
char indir[MAXDIR];
char infile[MAXFILE];
char inext[MAXEXT];

//////////////////////////////////////////////////////////////////////////////

void cleanup()
{
   if (fp) fclose(fp);
   if (search_value) free(search_value);
   if (work_buffer) free(work_buffer);
}

//////////////////////////////////////////////////////////////////////////////

int main(int argc, char *argv[])
{
   unsigned long f_size=0;

   if (argc != 3)
   {
      printf("usage: finddata [path] [data filename]\n");
      cleanup();
      exit (1);
   }

   fp = fopen(argv[2], "rb");

   if (fp == NULL)
   {
      printf("Unable to open: %s\n", argv[1]);
      cleanup();
      exit (1);
   }

   fseek(fp, 0, SEEK_END);
   f_size = ftell(fp);
   fseek(fp, 0, SEEK_SET);

   search_value = (unsigned char *)malloc(f_size);
   search_size = f_size;
   if (search_value == NULL)
   {
      printf("Unable to allocate memory\n");
      cleanup();
      exit (1);      
   }

   fread((void *)search_value, f_size, 1, fp);
   fclose(fp);

   // allocate work buffer
   if ((work_buffer = (unsigned char *)malloc(1000000)) == NULL)
   {
      printf("Unable to allocate memory\n");
      cleanup();
      exit(1);
   }

   // look for files
   SearchPath(argv[1]);

   // free all allocated memory
   cleanup();

   return 0;
}

//////////////////////////////////////////////////////////////////////////////

void SearchFile(char *filename, unsigned long size)
{
   unsigned long num_searches=0;
   unsigned long file_offset=0;
   unsigned long num_matches=0;
   unsigned long i, i2;

   fp = fopen(filename, "rb");

   if (fp == NULL)
   {
      printf("unable to open file\n");
      return;
   }
   else
   {
      for (;;)
      {
         memset(work_buffer, 0, 1000000);

         if (size > (1000000 + file_offset))
         {
            fread((void *)work_buffer, 1000000, 1, fp);
            num_searches = 1000000;
         }
         else
         {
            fread((void *)work_buffer, size, 1, fp);
            num_searches = size - file_offset;
         }

         // do comparisons
         for (i = 0; i < num_searches; i++)
         {
            for (i2 = 0; i2 < search_size; i2++)
            {
               if (work_buffer[i + i2] != search_value[i2])
               {
                  break;
               }

               if (i2 == search_size - 1)
               {
                  printf("\nfound a match at: %08x", file_offset + i);
                  num_matches += 1;
               }
            }
         }
          
         // if end of buffer and haven't grabbed everything yet, continue to
         // grab more and check until file completely read
         if (num_searches != 1000000)
            break;
         else
            file_offset += 1000000;
      }

      if (num_matches == 0)
         printf("no matches found.\n");
      else
         printf("\n");

      fclose(fp);

   }
}

//////////////////////////////////////////////////////////////////////////////

void SearchDir(char *dir)
{
   HANDLE hFind;
   WIN32_FIND_DATA result;
   char wildcard[MAXPATH];
   int done=0;

   printf("Searching %s\n", dir);

   // set wildcard string
   _splitpath (dir, indrive, indir, infile, inext);
   _makepath (wildcard, indrive, indir, "*.*", "");

   // start search
   if ((hFind = FindFirstFile(wildcard, &result)) != INVALID_HANDLE_VALUE)
   {
      if (!(result.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
      {
         _makepath (outpath, indrive, indir, result.cFileName, "");
         printf("Searching %s...", outpath);
         SearchFile(outpath, result.nFileSizeLow);
      }
   }
   else
   {
      printf("No files found\n");
      cleanup();
      exit (1);
   }

   while (!done)
   {
      if (FindNextFile(hFind, &result))
      {
         if (!(result.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
         {
            _makepath (outpath, indrive, indir, result.cFileName, "");
            printf("Searching %s...", outpath);
            SearchFile(outpath, result.nFileSizeLow);
         }
      }
      else
      {
         done = 1;
      }
   }

   FindClose(hFind);
}

//////////////////////////////////////////////////////////////////////////////

void SearchPath(char *path)
{
   HANDLE FindDir[32];
   char dirpath[MAX_PATH];
   unsigned char depth_counter=0;
   char subdir[MAX_PATH];
   unsigned long i;
                       
   _splitpath (path, indrive, indir, infile, inext);
   _makepath (dirpath, indrive, indir, infile, inext);

   if (dirpath[strlen(path)] != '\\')
   {
      strcat(dirpath, "\\");
   }

   // search base directory first
   SearchDir(dirpath);

   for (i = 0; i < 32; i++)
      FindDir[i] = INVALID_HANDLE_VALUE;

   printf("Next directory to search = %s\n", dirpath);

   for (;;)
   {
      if ((FindDir[depth_counter] = GrabSubDirectory(dirpath, FindDir[depth_counter], subdir)) == INVALID_HANDLE_VALUE)
      {
         if (depth_counter != 0) depth_counter--;
         else break;

         // truncate dirpath to directory below here
         TruncateLastDir(dirpath);
      }
      else
      {
         strcat(dirpath, subdir);
         strcat(dirpath, "\\");
         depth_counter++;

         SearchDir(dirpath);
      }
   }
}

//////////////////////////////////////////////////////////////////////////////

HANDLE GrabSubDirectory(char *path, HANDLE hFind, char *subdir)
{
   WIN32_FIND_DATA result;
   char wildcard[MAX_PATH];

   _splitpath (path, indrive, indir, infile, inext);
   _makepath (wildcard, indrive, indir, "*.*", "");

   // grab the next sub-directory(keep trying until I have one)
   for (;;)
   {
      if (hFind == INVALID_HANDLE_VALUE)
      {
         if ((hFind = FindFirstFile(wildcard, &result)) == INVALID_HANDLE_VALUE)
         {
            printf("FindFirstFile fail\n");
            return INVALID_HANDLE_VALUE;
         }
      }
      else
      {
         if (FindNextFile(hFind, &result) == FALSE)
         {
            FindClose(hFind);
            return INVALID_HANDLE_VALUE;
         }
      }

      if (result.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY &&
          strcmp(result.cFileName, ".") != 0 &&
          strcmp(result.cFileName, "..") != 0)                
      {
         strcpy(subdir, result.cFileName);
         break;
      }
      else
      {
//         printf("Found non-dir: %s\n", result.cFileName);
      }
   }

   return hFind;
}

//////////////////////////////////////////////////////////////////////////////

void TruncateLastDir(char *buffer)
{
   char *p;
   char *p2;

   p = strrchr(buffer, '\\');
   p[0] = 0x00; // set new end

   p2 = strrchr(buffer, '\\');
   p2[1] = 0x00; // set new end
}

//////////////////////////////////////////////////////////////////////////////

