/*******************************************************************************
  Saturn Dream Editor - a Saturn bitmap/tile viewer/editor

  (c) Copyright 2002-2003 Theo Berkau(cwx@softhome.net)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*******************************************************************************/
#ifndef WINVER
#define WINVER 0x500
#endif

#ifndef _WIN32_WINNT
#define _WIN32_WINNT 0x500
#endif

#include <allegro.h>
#include <winalleg.h>
#include <commctrl.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "main.h"
#include "resource.h"

static char szAppName[] = PROG_NAME;
static char szClassName[] = PROG_NAME;
HINSTANCE g_hInstance;

#undef DEBUG

#define TIMER_ID 1

#define MAX_WIDTH 640
#define MAX_HEIGHT 480
//#define MAX_WIDTH 512
//#define MAX_HEIGHT 256

//////////////////////////////////////////////////////////////////////////////

#define MAX_FILEPATH    256

static char strPath[MAX_PATH] = "";
OPENFILENAME file;
OPENFILENAME palette_file;
OPENFILENAME bitmap_file;
char gszOpenName[MAX_FILEPATH];
char gszInitialDir[MAX_FILEPATH];
char gszPaletteOpenName[MAX_FILEPATH];
char gszPaletteInitialDir[MAX_FILEPATH];
char gszBitmapOpenName[MAX_FILEPATH];
char gszBitmapInitialDir[MAX_FILEPATH];

int file_loaded=FALSE;

//////////////////////////////////////////////////////////////////////////////

FILE *fp;
unsigned long f_size;
unsigned long f_size2;
unsigned long f_offset=0;
unsigned long f_offset2=0;
unsigned short tile_width=MAX_WIDTH, tile_height=MAX_HEIGHT;
unsigned short tile_width2=MAX_WIDTH, tile_height2=MAX_HEIGHT;
unsigned char bpp=4;
unsigned char bpp2=4;
unsigned char bswap = 0;
unsigned char bswap2 = 0;
unsigned char num_col=1;
unsigned char num_col2=1;
unsigned char num_row=0;
unsigned char num_row2=0;
unsigned char usr_col=1;
unsigned char usr_col2=1;
unsigned char auto_col=1;
unsigned char auto_col2=1;
unsigned char *work_buffer;
unsigned char *work_buffer2;
unsigned char *scratch_pad;

bool used_cancel=FALSE;
bool editmode=FALSE;

BITMAP *double_buffer;
PALETTE user_palette;

//////////////////////////////////////////////////////////////////////////////

void BufferToBitmap1bpp(unsigned char *buffer, BITMAP *bitmap, unsigned short x, unsigned short y);
void BufferToBitmap2bpp(unsigned char *buffer, BITMAP *bitmap, unsigned short x, unsigned short y);
void BufferToBitmap4bpp(unsigned char *buffer, BITMAP *bitmap, unsigned short x, unsigned short y);
void BufferToBitmap8bpp(unsigned char *buffer, BITMAP *bitmap, unsigned short x, unsigned short y);
void BufferToBitmap16bpp(unsigned char *buffer, BITMAP *bitmap, unsigned short x, unsigned short y);
void BufferToBitmap24bpp(unsigned char *buffer, BITMAP *bitmap, unsigned short x, unsigned short y);

void DrawPage (HWND hWnd);
void UpdateScreen();

//////////////////////////////////////////////////////////////////////////////

unsigned long collength()
{
   if (bpp <= 8)
   {
      return (tile_width * tile_height / (8 / bpp));
   }
   else
   {
      if (bpp == 24)
         return (tile_width * tile_height * 4);
      else
         return (tile_width * tile_height * (bpp / 8));
   }
}

//////////////////////////////////////////////////////////////////////////////

unsigned long rowlength()
{
   return (collength() * num_col);
}

//////////////////////////////////////////////////////////////////////////////

unsigned long pagelength()
{
   return (rowlength() * num_row);   
}

//////////////////////////////////////////////////////////////////////////////

LRESULT CALLBACK WindowProc(HWND hWnd,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
   HRESULT hRet;
   HMENU hMenu;

   switch (uMsg)
   {
      case WM_COMMAND:
      {
         switch (LOWORD(wParam))
         {
            case IDM_LOAD:
            {
               if (GetOpenFileName(&file))
               {
                   //first make sure buffer is empty

                   if (work_buffer) free (work_buffer);

                   //Load file here

                   fp = fopen(gszOpenName,"rb");

                   if (fp == NULL)
                   {
                      MessageBox (hWnd, "Error loading file", "Error",
                                  MB_OK | MB_ICONINFORMATION);
                      file_loaded = FALSE;
                   }
                   else
                   {
                      // reset the offset or face the consequences!
                      f_offset = 0;

                      f_size = filelength(fileno(fp));

                      work_buffer = (unsigned char *)malloc(f_size);

                      if (work_buffer == NULL)
                      {
                         MessageBox (hWnd, "Error loading file - not enough memory", "Error",
                         MB_OK | MB_ICONINFORMATION);

                         file_loaded = FALSE;
                      }
                      else
                      {
                         fread(work_buffer, 1, filelength(fileno(fp)), fp);

                         file_loaded = TRUE;
                      }

                      fclose (fp);

                      DrawPage (hWnd);
                      UpdateScreen();

//                      file.lpstrTitle = gszOpenName;

                      hMenu = GetMenu(hWnd);
                      EnableMenuItem(hMenu, IDM_SAVE, MF_ENABLED);
                      EnableMenuItem(hMenu, IDM_SAVEAS, MF_ENABLED);
                      EnableMenuItem(hMenu, IDM_LOADPALETTE, MF_ENABLED);
                      EnableMenuItem(hMenu, IDM_IMPORTBMP, MF_ENABLED);
                      EnableMenuItem(hMenu, IDM_EXPORTBMP, MF_ENABLED);
                   }
               }
               break;
            }
            case IDM_SAVE:
            {
               fp = fopen(gszOpenName,"wb");

               if (fp == NULL)
               {
                  MessageBox (hWnd, "Error saving file", "Error",
                              MB_OK | MB_ICONINFORMATION);
               }
               else
               {
                  fwrite(work_buffer, 1, f_size, fp);
                  fclose (fp);
               }

               break;
            }
            case IDM_SAVEAS:
            {
               if (GetSaveFileName(&file))
               {
                   //first make sure buffer is empty

                   //Load file here

                   fp = fopen(gszOpenName,"wb");

                   if (fp == NULL)
                   {
                      MessageBox (hWnd, "Error saving file", "Error",
                                  MB_OK | MB_ICONINFORMATION);
                   }
                   else
                   {
                      fwrite(work_buffer, 1, f_size, fp);
                      fclose (fp);

//                      MessageBox (hWnd, "File saved successfully!", "Notice",
//                                  MB_OK | MB_ICONINFORMATION);
                   }

               }
               break;
            }
            case IDM_LOADPALETTE:
            {
               if (GetOpenFileName(&palette_file))
               {
                   //Load file here

                   fp = fopen(gszPaletteOpenName,"rb");

                   if (fp == NULL)
                   {
                      MessageBox (hWnd, "Error loading file", "Error",
                                  MB_OK | MB_ICONINFORMATION);
                   }
                   else
                   {
                      if (filelength(fileno(fp)) != 768)
                      {
                         MessageBox (hWnd, "Not a palette file", "Error",
                                     MB_OK | MB_ICONINFORMATION);
                      }
                      else
                      {
                         int i;

                         for (i = 0; i < 256; i++)
                         {
                            fread((void *)&user_palette[i].r, 1, 1, fp);
                            fread((void *)&user_palette[i].g, 1, 1, fp);
                            fread((void *)&user_palette[i].b, 1, 1, fp);

                            user_palette[i].r >>= 2;
                            user_palette[i].g >>= 2;
                            user_palette[i].b >>= 2;
                         }

                         DrawPage (hWnd);
                         UpdateScreen();
                      }

                      fclose (fp);

                      // If loading succesful, check to see if other files
                      // have already been loaded. If so, start showing output

//                      palette_file.lpstrTitle = gszPaletteOpenName;
                   }
               }
               break;
            }
            case IDM_IMPORTBMP:
            {
               BITMAPFILEHEADER hdr;
               BITMAPINFOHEADER bmpinfo;

               if (GetOpenFileName(&bitmap_file))
               {
                  //load file here

                  fp = fopen(gszBitmapOpenName,"rb");

                  if (fp == NULL)
                  {
                     MessageBox (hWnd, "Error opening file", "Error",
                                 MB_OK | MB_ICONINFORMATION);

                     return 0L;
                  }

                  memset(&hdr, 0, sizeof(BITMAPFILEHEADER));
                  memset(&bmpinfo, 0, sizeof(BITMAPFILEHEADER));

                  fread((void *)&hdr, sizeof(BITMAPFILEHEADER), 1, fp);

                  if (hdr.bfType != 0x4D42)
                  {
                     MessageBox (hWnd, "Not a BMP file", "Error",
                                 MB_OK | MB_ICONINFORMATION);
                     fclose (fp);
                     return 0L;
                  }

                  if (hdr.bfSize != filelength(fileno(fp)))
                  {
                     MessageBox (hWnd, "BMP header error: Size Mismatch", "Error",
                                 MB_OK | MB_ICONINFORMATION);                           

                     fclose (fp);
                     return 0L;
                  }

                  fread((void *)&bmpinfo.biSize, 1, 4, fp);

                  switch(bmpinfo.biSize)
                  {
                     case sizeof(BITMAPINFOHEADER):
                     {
                        fseek(fp, -4, SEEK_CUR);

                        fread((void *)&bmpinfo, sizeof(BITMAPINFOHEADER), 1, fp);
                        break;
                     }
                     default:
                     {
                        MessageBox (hWnd, "BMP header error: Invalid Bitmap info header", "Error",
                                    MB_OK | MB_ICONINFORMATION);                           
                        fclose (fp);
                        return 0L;
                     }
                  }

                  if (bmpinfo.biCompression != BI_RGB)
                  {
                     MessageBox (hWnd, "BMP header error: Unsupported Compression", "Error",
                                 MB_OK | MB_ICONINFORMATION);                           

                     fclose (fp);
                     return 0L;
                  }


                  switch (bmpinfo.biBitCount)
                  {
                     case 1: 
                     case 4:
                     default: break;
                  }

                  fclose (fp);
               }

               break;
            }
            case IDM_EXPORTBMP:
            {
               BITMAPFILEHEADER hdr;
               BITMAPINFOHEADER bmpinfo;
               FILE *fp;
               int i, i2, i3, i4;

               if (GetSaveFileName(&bitmap_file))
               {
                  //Save file here

                  memset(&hdr, 0, sizeof(BITMAPFILEHEADER));
                  memset(&bmpinfo, 0, sizeof(BITMAPINFOHEADER));

                  bmpinfo.biSize = sizeof(BITMAPINFOHEADER);
                  bmpinfo.biWidth = (tile_width * num_col);
                  bmpinfo.biHeight = 0 - (tile_height * num_row);
                  bmpinfo.biPlanes = 1;

                  if (bpp != 2)
                     bmpinfo.biBitCount = bpp;
                  else
                     bmpinfo.biBitCount = 4; // convert 2bpp to 4bpp

                  bmpinfo.biCompression = BI_RGB;
                  if (bpp != 2)
                     bmpinfo.biSizeImage = pagelength();
                  else
                     bmpinfo.biSizeImage = pagelength() * 2; 

                  bmpinfo.biXPelsPerMeter = 0;
                  bmpinfo.biYPelsPerMeter = 0;
                  if (bpp < 16)
                     bmpinfo.biClrUsed = (1 << bpp);
                  bmpinfo.biClrImportant = 0;

                  hdr.bfType = 0x4d42;        // 0x42 = "B" 0x4d = "M"

                  // Compute the size of the entire file.
                  if (bpp == 1 && ((num_col * tile_width) % 32) != 0)
                  {
                     hdr.bfSize = (DWORD) (sizeof(BITMAPFILEHEADER) +
                                  sizeof(BITMAPINFOHEADER) +
                                  (sizeof(RGBQUAD) * bmpinfo.biClrUsed) +
                                  bmpinfo.biSizeImage +
                                  (((32 - ((num_col * tile_width) % 32)) / 8) * tile_height * num_row));
                  }
                  else if (bpp == 24)
                  {
                     hdr.bfSize = (DWORD) (sizeof(BITMAPFILEHEADER) +
                                  sizeof(BITMAPINFOHEADER) +
                                  (sizeof(RGBQUAD) * bmpinfo.biClrUsed) +
                                  (bmpinfo.biSizeImage * 0.75)); 
                  }
                  else
                  {
                     hdr.bfSize = (DWORD) (sizeof(BITMAPFILEHEADER) +
                                  sizeof(BITMAPINFOHEADER) +
                                  (sizeof(RGBQUAD) * bmpinfo.biClrUsed) +
                                  bmpinfo.biSizeImage); 
                  }

                  hdr.bfReserved1 = 0;
                  hdr.bfReserved2 = 0; 
                  hdr.bfOffBits = (DWORD) sizeof(BITMAPFILEHEADER) +
                                  bmpinfo.biSize +
                                  (sizeof(RGBQUAD) * bmpinfo.biClrUsed);

                  fp = fopen(gszBitmapOpenName,"wb");

                  if (fp == NULL)
                  {
                     MessageBox (hWnd, "Error saving file", "Error",
                                 MB_OK | MB_ICONINFORMATION);                    
                  }
                  else
                  {
                     fwrite((void *)&hdr, sizeof(BITMAPFILEHEADER), 1, fp);
                     fwrite((void *)&bmpinfo, sizeof(BITMAPINFOHEADER), 1, fp);
                     // write palette here

                     if (bmpinfo.biClrUsed != 0)
                     {
                        for (i = 0; i < bmpinfo.biClrUsed; i++)
                        {
                           unsigned long pal=0;
                           pal = ((user_palette[i].b << 2)) +
                                 ((user_palette[i].g << 2) << 8) +
                                 ((user_palette[i].r << 2) << 16);

                           fwrite((void *)&pal, 1, 4, fp);
                        }
                     }

                     switch (bpp)
                     {
                        case 1:
                        {
                           for (i4 = 0; i4 < num_row; i4++)
                           {
                              for (i3 = 0; i3 < tile_height; i3++)
                              {
                                 for (i2 = 0; i2 < num_col; i2++)
                                 {
                                    for (i = 0; i < tile_width; i++)
                                    {
                                       unsigned char pixel=0;
                                       unsigned char pixel2=0;

                                       if ((i % 8) == 0)
                                       {
                                          pixel = work_buffer[f_offset + (((i4 * tile_width * tile_height * num_col) + (i3 * tile_width) + (i2 * tile_width * tile_height) + i) / 8)];

                                          fwrite((void *)&pixel, 1, 1, fp);
                                       }
                                    }
                                 }

                                 if (((num_col * tile_width) % 32) != 0)
                                 {
                                    unsigned char n=0;

                                    for (n = 0; n < (((num_col * tile_width) % 32) / 8); n++)
                                    {
                                       fputc(0x00, fp);
                                    }
                                 }
                              }
                           }

                           break;
                        }
                        case 2:
                        {
                           for (i4 = 0; i4 < num_row; i4++)
                           {
                              for (i3 = 0; i3 < tile_height; i3++)
                              {
                                 for (i2 = 0; i2 < num_col; i2++)
                                 {
                                    for (i = 0; i < tile_width; i++)
                                    {
                                       unsigned char pixel=0;
                                       unsigned char pixel2=0;

                                       if ((i % 4) == 0)
                                       {
                                          pixel = work_buffer[f_offset + (((i4 * tile_width * tile_height * num_col) + (i3 * tile_width) + (i2 * tile_width * tile_height) + i) / 4)];

                                          pixel2 = (pixel & 0x0F);
                                          pixel = (pixel & 0xF0) >> 4;

                                          pixel = ((pixel & 0x0C) << 2) + (pixel & 0x03);
                                          pixel2 = ((pixel2 & 0x0C) << 2) + (pixel2 & 0x03);

                                          fwrite((void *)&pixel, 1, 1, fp);
                                          fwrite((void *)&pixel2, 1, 1, fp);
                                       }
                                    }
                                 }
                              }
                           }

                           break;
                        }
                        case 4:
                        {
                           for (i4 = 0; i4 < num_row; i4++)
                           {
                              for (i3 = 0; i3 < tile_height; i3++)
                              {
                                 for (i2 = 0; i2 < num_col; i2++)
                                 {
                                    for (i = 0; i < tile_width; i++)
                                    {
                                       unsigned char pixel=0;

                                       if ((i % 2) == 0)
                                       {
                                          pixel = work_buffer[f_offset + (((i4 * tile_width * tile_height * num_col) + (i3 * tile_width) + (i2 * tile_width * tile_height) + i) / 2)];

                                          fwrite((void *)&pixel, 1, 1, fp);
                                       }
                                    }
                                 }
                              }
                           }

                           break;
                        }
                        case 8:
                        {
                           for (i4 = 0; i4 < num_row; i4++)
                           {
                              for (i3 = 0; i3 < tile_height; i3++)
                              {
                                 for (i2 = 0; i2 < num_col; i2++)
                                 {
                                    for (i = 0; i < tile_width; i++)
                                    {
                                       unsigned char pixel=0;

                                       pixel = work_buffer[f_offset + (i4 * tile_width * tile_height * num_col) + (i3 * tile_width) + (i2 * tile_width * tile_height) + i];
 
                                       fwrite((void *)&pixel, 1, 1, fp);
                                    }
                                 }
                              }
                           }

                           break;
                        }
                        case 16:
                        {
                           for (i4 = 0; i4 < num_row; i4++)
                           {
                              for (i3 = 0; i3 < tile_height; i3++)
                              {
                                 for (i2 = 0; i2 < num_col; i2++)
                                 {
                                    for (i = 0; i < tile_width; i++)
                                    {
                                       unsigned short pixel=0;

                                       pixel = (work_buffer[f_offset + (((i4 * tile_width * tile_height * num_col) + (i3 * tile_width) + (i2 * tile_width * tile_height) + i) * 2)] << 8) +
                                                work_buffer[f_offset + (((i4 * tile_width * tile_height * num_col) + (i3 * tile_width) + (i2 * tile_width * tile_height) + i) * 2) + 1];

                                       pixel = ((pixel & 0x7C00) >> 10) + (pixel & 0x03E0) + ((pixel & 0x001F) << 10);

                                       fwrite((void *)&pixel, 2, 1, fp);
                                    }
                                 }
                              }
                           }

                           break;
                        }
                        case 24:
                        {
                           for (i4 = 0; i4 < num_row; i4++)
                           {
                              for (i3 = 0; i3 < tile_height; i3++)
                              {
                                 for (i2 = 0; i2 < num_col; i2++)
                                 {
                                    for (i = 0; i < tile_width; i++)
                                    {
                                       unsigned char r=0, g=0, b=0;

                                       r = work_buffer[f_offset + (((i4 * tile_width * tile_height * num_col) + (i3 * tile_width) + (i2 * tile_width * tile_height) + i) * 4) + 1];
                                       g = work_buffer[f_offset + (((i4 * tile_width * tile_height * num_col) + (i3 * tile_width) + (i2 * tile_width * tile_height) + i) * 4) + 2];
                                       b = work_buffer[f_offset + (((i4 * tile_width * tile_height * num_col) + (i3 * tile_width) + (i2 * tile_width * tile_height) + i) * 4) + 3];

                                       fwrite((void *)&r, 1, 1, fp);
                                       fwrite((void *)&g, 1, 1, fp);
                                       fwrite((void *)&b, 1, 1, fp);
                                    }
                                 }
                              }
                           }

                           break;
                        }
                        default: break;
                     }

                     fclose (fp);
                  }
               }

               break;
            }
            case IDM_SETTINGS:
            {
               DialogBox(g_hInstance, "SettingsDlg", hWnd, (DLGPROC)SettingsDlgProc);
               break;
            }
            case IDM_ABOUT:
            {
               DialogBox(g_hInstance, "AboutDlg", hWnd, (DLGPROC)AboutDlgProc);
               break;
            }
            case IDM_EXIT:
               PostMessage(hWnd, WM_CLOSE, 0, 0);
               break;
         }
         return 0L;
      }
      case WM_GETMINMAXINFO:
      {
         return 0L;
      }
      case WM_MOVE:
      {
         return 0L;
      }
      case WM_PAINT:
      {
         PAINTSTRUCT ps;

         BeginPaint(hWnd, &ps);
         EndPaint(hWnd, &ps);
         return 0L;
      }
      case WM_SIZE:
      {
         return 0L;
      }
      case WM_KEYDOWN:
      {
         switch (wParam)
         {
            case VK_SUBTRACT:
            {
               if (f_offset > 0)
                  f_offset -= 1;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case VK_ADD:
            {
//               if ((f_offset + 1 + pagelength()) <= f_size)
               f_offset += 1;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case VK_UP:
            {
               if (f_offset >= rowlength())
                  f_offset -= rowlength();
               else 
                  f_offset = 0;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case VK_DOWN:
            {
//               if ((f_offset + rowlength() + pagelength()) < f_size)
                  f_offset += rowlength();
//               else
//               {
//                  if (f_size >= pagelength())
//                     f_offset = f_size - pagelength();
//               }

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case VK_LEFT:
            {
               if (f_offset >= collength())
                  f_offset -= collength();
               else 
                  f_offset = 0;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case VK_RIGHT:
            {
//               if ((f_offset + collength() + pagelength()) < f_size)
                  f_offset += collength();
//               else
//               {
//                  if (f_size >= pagelength())
//                     f_offset = f_size - pagelength();
//               }

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case VK_PRIOR: // PAGE UP
            {
               if (f_offset >= pagelength())
               {
                  f_offset -= pagelength();
               }
               else
                  f_offset = 0;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case VK_NEXT: // PAGE DOWN
            {
//               if ((f_offset + pagelength() + pagelength()) < f_size)
                  f_offset += pagelength();
//               else
//               {
//                  if (f_size >= pagelength())
//                     f_offset = f_size - pagelength();
//               }

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case VK_HOME:
            {
               f_offset = 0;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case VK_END:
            {
               if (f_size >= pagelength())
                  f_offset = f_size - pagelength();

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case 0x30: // 0 key
            {
               // Custom
               break;
            }
            case 0x31: // 1 key
            {
               bpp = 1;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case 0x32: // 2 key
            {
               bpp = 2;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case 0x33: // 3 key
            {
               bpp = 4;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case 0x34: // 4 key
            {
               bpp = 8;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case 0x35: // 5 key
            {
               bpp = 16;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case 0x36: // 6 key
            {
               // 24 BPP
               bpp = 24;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case 0x41: // A key
            {
               if (auto_col)
               {
                  auto_col = 0;
                  usr_col = num_col;

                  DrawPage (hWnd);
                  UpdateScreen();
               }
               else
               {
                  auto_col = 1;

                  DrawPage (hWnd);
                  UpdateScreen();
               }
                  
               break;
            }
            case 0x51: // Q key
            {
               if (auto_col == 0 && usr_col > 0)
               {
                  usr_col--;

                  DrawPage (hWnd);
                  UpdateScreen();
               }

               break;
            }
            case 0x57: // W key
            {
               if (auto_col == 0 && (usr_col + 1) <= (MAX_WIDTH / tile_width))
               {
                  usr_col++;

                  DrawPage (hWnd);
                  UpdateScreen();
               }

               break;
            }
            case 0x49: // I key
            {
               if (tile_height > 1)
                  tile_height -= 1;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case 0x4B: // K key
            {
               if (tile_height < MAX_HEIGHT)
                  tile_height += 1;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case 0x4A: // J key
            {
               if (tile_width > 8)
//                  tile_width -= 8;
                  tile_width -= 1;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            case 0x4C: // L key
            {
               if (tile_width < MAX_WIDTH)
//                  tile_width += 8;
                  tile_width += 1;

               DrawPage (hWnd);
               UpdateScreen();

               break;
            }
            default: break;
         }
         return 0L;
      }
      case WM_MOUSEWHEEL:
      {
         int xPos=0, yPos=0;
         short zDelta=0;
         RECT rcClient;
         POINT ptClientUL;
         POINT ptClientLR;

         // grab mouse position
         xPos = (int)LOWORD(lParam);
         yPos = (int)HIWORD(lParam);

         // get window information
         GetClientRect(hWnd, &rcClient);
         ptClientUL.x = rcClient.left;
         ptClientUL.y = rcClient.top;

         ptClientLR.x = rcClient.right + 1;
         ptClientLR.y = rcClient.bottom + 1; 

         // convert to screen coordinates
         ClientToScreen(hWnd, &ptClientUL);
         ClientToScreen(hWnd, &ptClientLR);

         // Check to make sure the x and y Position of the mouse is over the
         // editing area.

         if (xPos >= ptClientUL.x && xPos < ptClientLR.x &&
             yPos >= ptClientUL.y && yPos < ptClientLR.y)
         {
            zDelta = (short)HIWORD(wParam);

            if (LOWORD(wParam) == MK_CONTROL)
            {
               // Left/Right one Tile Col

               if (zDelta > 0)
               {
                  if (f_offset >= collength())
                     f_offset -= collength();
                  else 
                     f_offset = 0;

                  DrawPage (hWnd);
                  UpdateScreen();
               }
               else if (zDelta < 0)
               {
                  if ((f_offset + collength() + pagelength()) < f_size)
                     f_offset += collength();
                  else
                  {
                     if (f_size >= pagelength())
                        f_offset = f_size - pagelength();
                  }

                  DrawPage (hWnd);
                  UpdateScreen();
               }
            }
            else if (LOWORD(wParam) == MK_SHIFT)
            {
               // Offset by 1 byte

               if (zDelta > 0)
               {
                  if (f_offset > 0)
                     f_offset -= 1;

                  DrawPage (hWnd);
                  UpdateScreen();
               }
               else if (zDelta < 0)
               {
                  if ((f_offset + 1 + pagelength()) <= f_size)
                  f_offset += 1;

                  DrawPage (hWnd);
                  UpdateScreen();
               }
            }
            else
            {
               // Up/Down one Tile Row

               if (zDelta > 0)
               {
                  if (f_offset >= rowlength())
                     f_offset -= rowlength();
                  else 
                     f_offset = 0;

                  DrawPage (hWnd);
                  UpdateScreen();
               }
               else if (zDelta < 0)
               {
                  if ((f_offset + rowlength() + pagelength()) < f_size)
                     f_offset += rowlength();
                  else
                  {
                     if (f_size >= pagelength())
                        f_offset = f_size - pagelength();
                  }

                  DrawPage (hWnd);
                  UpdateScreen();
               }
            }
         }

         return 0L;
      }

      case WM_DESTROY:
      {
         PostQuitMessage(0);

         if (work_buffer)
            free(work_buffer);
         if (work_buffer2)
            free(work_buffer2);
         if (scratch_pad)
            free(scratch_pad);

         destroy_bitmap(double_buffer);

         return 0L;
      }

    }

    return DefWindowProc(hWnd, uMsg, wParam, lParam);
}

//////////////////////////////////////////////////////////////////////////////

int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine, int nCmdShow)
{
   WNDCLASS                    MyWndClass;
   HRESULT                     hRet;
   HWND                        hWnd;
   int                         cx, cy;
   MSG                         msg;
   int                         i, i2;
   char INIFileName[MAX_PATH];
   DWORD INIFileNameSize=0;
   char *pINIFileName;
   char tempstr[MAX_PATH];

   // Set up and register window class
   MyWndClass.style = CS_HREDRAW | CS_VREDRAW;
   MyWndClass.lpfnWndProc = (WNDPROC) WindowProc;
   MyWndClass.cbClsExtra = 0;
   MyWndClass.cbWndExtra = sizeof(DWORD);
   MyWndClass.hInstance = hInstance;
    MyWndClass.hIcon = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ICON));
   MyWndClass.hCursor = LoadCursor(NULL, IDC_ARROW);
//   MyWndClass.hbrBackground = (HBRUSH)COLOR_WINDOW;//(HBRUSH) GetStockObject(GRAY_BRUSH);
   MyWndClass.hbrBackground = (HBRUSH) GetStockObject(BLACK_BRUSH);
   MyWndClass.lpszMenuName = MAKEINTRESOURCE(IDR_MENU);
   MyWndClass.lpszClassName = szClassName;

   if (!RegisterClass(&MyWndClass))
       return FALSE;

//    switch (GetSystemMetrics(SM_CXSCREEN))
//    {
//       case 640:
//       {
//          window_w = 320;
//          window_h = 240;
//          scale=1;
//          break;
//       }
//       case 800:
//       case 848:
//       {
//          window_w = 640;
//          window_h = 480;
//          scale=2;
//          break;
//       }
//       case 1024:
//       case 1152:
//       case 1280:
//       {
//          window_w = 960;
//          window_h = 720;
//          scale=3;
//          break;
//       }
//       case 1600:
//       {
//          window_w = 1280;
//          window_h = 960;
//          scale=4;
//          break;
//       }
//       case 1800:
//       case 1856:
//       {
//          window_w = 1600;
//          window_h = 1200;
//          scale=5;
//          break;
//       }
//       default:
//       {
//          window_w = 640;
//          window_h = 480;
//          scale=1;
//          break;
//       }
//    }

   // get program pathname
   INIFileNameSize = GetModuleFileName(hInstance, INIFileName, MAX_PATH);

   // set pointer to start of extension
   pINIFileName = INIFileName + INIFileNameSize - 4;

   // replace .exe with .ini
   sprintf(pINIFileName, ".ini\0");

//   if (GetPrivateProfileInt("GeneralSection","RememberSettings", 0, INIFileName))
   {
      GetPrivateProfileString("GeneralSection", "TilePath", "", gszInitialDir, MAX_PATH, INIFileName);
      GetPrivateProfileString("GeneralSection", "PalettePath", "", gszPaletteInitialDir, MAX_PATH, INIFileName);
      GetPrivateProfileString("GeneralSection", "BitmapPath", "", gszBitmapInitialDir, MAX_PATH, INIFileName);

      bpp = GetPrivateProfileInt("VisualsSection","DefaultBPPMode", 4, INIFileName);

      tile_width = GetPrivateProfileInt("VisualsSection","DefaultTileWidth", MAX_WIDTH, INIFileName);
      tile_height = GetPrivateProfileInt("VisualsSection","DefaultTileHeight", MAX_HEIGHT, INIFileName);

      auto_col = GetPrivateProfileInt("VisualsSection","AutoAdjustColumns", TRUE, INIFileName);
      num_col = GetPrivateProfileInt("VisualsSection","DefaultNumberOfColumns", 1, INIFileName);
      usr_col = num_col;
      num_row = GetPrivateProfileInt("VisualsSection","DefaultNumberOfRows", 1, INIFileName);
   }

   cx = 640 + GetSystemMetrics(SM_CXSIZEFRAME) * 2;
   cy = 480 + (GetSystemMetrics(SM_CYSIZEFRAME) * 2) + GetSystemMetrics(SM_CYMENU) + GetSystemMetrics(SM_CYCAPTION);

   // Create a window
   hWnd = CreateWindow(szClassName,        // class
                       szAppName,          // caption
                       WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU |
                       WS_THICKFRAME | WS_MINIMIZEBOX |  // style
                       WS_CLIPCHILDREN,  
                       CW_USEDEFAULT,      // x pos
                       CW_USEDEFAULT,      // y pos
                       cx,        // width
                       cy,       // height
                       HWND_DESKTOP,       // parent window
                       NULL,               // menu 
                       hInstance,          // instance
                       NULL);              // parms

   if (!hWnd)
       return FALSE;

   ShowWindow(hWnd, nCmdShow);
   UpdateWindow(hWnd);

   set_color_depth(24);
   win_set_window(hWnd);
   allegro_init();

//   if (set_gfx_mode(GFX_AUTODETECT_WINDOWED, 640, 480, 0, 0) != 0)
//   if (set_gfx_mode(GFX_GDI, 640, 480, 0, 0) != 0)
   if (set_gfx_mode(GFX_DIRECTX_WIN, 640, 480, 0, 0) != 0)
   {
      MessageBox (hWnd, "Error Setting up Graphics", "Error",
                  MB_OK | MB_ICONINFORMATION);
      return FALSE;
   }

   if (set_display_switch_mode(SWITCH_BACKGROUND) == -1)
   {
      MessageBox (hWnd, "Error setting up display switch mode", "Error",
                  MB_OK | MB_ICONINFORMATION);
      return FALSE;
   }

   if (install_timer() != 0)
   {
      MessageBox (hWnd, "Error Setting up Timer API", "Error",
                  MB_OK | MB_ICONINFORMATION);
   }
   else if (install_mouse() == -1)
   {
      MessageBox (hWnd, "Error Setting up Mouse API", "Error",
                  MB_OK | MB_ICONINFORMATION);
      return FALSE;
   }

   double_buffer = create_bitmap (MAX_WIDTH, MAX_HEIGHT);

   if (double_buffer == NULL)
   {
      MessageBox (hWnd, "Error creating Work Bitmap", "Error",
                  MB_OK | MB_ICONINFORMATION);

      return FALSE;
   }

   scratch_pad = (unsigned char *)malloc(MAX_WIDTH * MAX_HEIGHT * 4);

   if (scratch_pad == NULL)
   {
      MessageBox (hWnd, "Error creating scratch pad", "Error",
                  MB_OK | MB_ICONINFORMATION);

      destroy_bitmap(double_buffer);
      return FALSE;
   }


   // copy default palette to user palette
   for (i = 0; i < 255; i++)
   {
      user_palette[i].r = default_palette[i].r;
      user_palette[i].g = default_palette[i].g;
      user_palette[i].b = default_palette[i].b;
   }

   memset(&file, 0, sizeof(OPENFILENAME));
   file.lStructSize = sizeof(OPENFILENAME);
   file.hwndOwner = hWnd;
   file.lpstrFilter = "All Files (*.*)\0*.*\0";
   file.lpstrFile = gszOpenName;
   file.nMaxFile = sizeof(gszOpenName);
   file.lpstrInitialDir = gszInitialDir;
   file.Flags = OFN_HIDEREADONLY;

   memset(&palette_file, 0, sizeof(OPENFILENAME));
   palette_file.lStructSize = sizeof(OPENFILENAME);
   palette_file.hwndOwner = hWnd;
   palette_file.lpstrFilter = "Palette Files (*.act)\0*.act\0All Files (*.*)\0*.*\0";
   palette_file.lpstrFile = gszPaletteOpenName;
   palette_file.nMaxFile = sizeof(gszPaletteOpenName);
   palette_file.lpstrInitialDir = gszPaletteInitialDir;
   palette_file.Flags = OFN_HIDEREADONLY;

   memset(&bitmap_file, 0, sizeof(OPENFILENAME));
   bitmap_file.lStructSize = sizeof(OPENFILENAME);
   bitmap_file.hwndOwner = hWnd;
   bitmap_file.lpstrFilter = "Bitmap Files (*.bmp)\0*.bmp\0All Files (*.*)\0*.*\0";
   bitmap_file.lpstrFile = gszBitmapOpenName;
   bitmap_file.nMaxFile = sizeof(gszPaletteOpenName);
   bitmap_file.lpstrInitialDir = gszBitmapInitialDir;
   bitmap_file.Flags = OFN_HIDEREADONLY;

   gszOpenName[0] = 0;
   gszPaletteOpenName[0] = 0;
   gszBitmapOpenName[0] = 0;

   g_hInstance = hInstance;

   while (GetMessage(&msg, NULL, NULL, NULL))
   {
      poll_mouse();

      if (editmode == 0)
      {
         if (mouse_b & 1)
         {
            // copy data from the file to the clipboard
         }
      }

      TranslateMessage(&msg);
      DispatchMessage(&msg);
   }

   // Save Settings to INI

   // write "RememberSettings" here

//   if (GetPrivateProfileInt("GeneralSection","RememberSettings", 0, INIFileName))
   {
      memset (tempstr, 0, MAX_PATH);
      memcpy (tempstr, file.lpstrFile, file.nFileOffset);
      WritePrivateProfileString("GeneralSection", "TilePath", tempstr, INIFileName);

      memset (tempstr, 0, MAX_PATH);
      memcpy (tempstr, palette_file.lpstrFile, palette_file.nFileOffset);
      WritePrivateProfileString("GeneralSection", "PalettePath", tempstr, INIFileName);

      memset (tempstr, 0, MAX_PATH);
      memcpy (tempstr, bitmap_file.lpstrFile, bitmap_file.nFileOffset);
      WritePrivateProfileString("GeneralSection", "BitmapPath", tempstr, INIFileName);

      WritePrivateProfileString("VisualsSection", "DefaultBPPMode", itoa(bpp,tempstr,10),INIFileName);

      WritePrivateProfileString("VisualsSection", "DefaultTileWidth", itoa(tile_width,tempstr,10),INIFileName);
      WritePrivateProfileString("VisualsSection", "DefaultTileHeight", itoa(tile_height,tempstr,10),INIFileName);

      WritePrivateProfileString("VisualsSection", "AutoAdjustColumns", itoa(auto_col,tempstr,10),INIFileName);
      WritePrivateProfileString("VisualsSection", "DefaultNumberOfColumns", itoa(num_col,tempstr,10),INIFileName);
      WritePrivateProfileString("VisualsSection", "DefaultNumberOfRows", itoa(num_row,tempstr,10),INIFileName);
   }


   return (msg.wParam);
}

//////////////////////////////////////////////////////////////////////////////

LRESULT CALLBACK AboutDlgProc(HWND hDlg, UINT uMsg, WPARAM wParam,
                              LPARAM lParam)
{
   switch (uMsg)
   {
      case WM_INITDIALOG:
      {
         return TRUE;
      }
      case WM_COMMAND:
      {
         switch (wParam)
         {
            case IDOK:
            {
               EndDialog(hDlg, TRUE);
               return TRUE;
            }
            default: break;
         }
         break;
      }
   }

   return FALSE;
}

//////////////////////////////////////////////////////////////////////////////

LRESULT CALLBACK SettingsDlgProc(HWND hDlg, UINT uMsg, WPARAM wParam,
                                 LPARAM lParam)
{
   switch (uMsg)
   {
      case WM_INITDIALOG:
      {
         return TRUE;
      }
      case WM_COMMAND:
      {
         switch (wParam)
         {
            case IDOK:
            {
               EndDialog(hDlg, TRUE);
               return TRUE;
            }
            case IDCANCEL:
            {
               EndDialog(hDlg, FALSE);
               return TRUE;
            }
            default: break;
         }
         break;
      }
   }

   return FALSE;
}

//////////////////////////////////////////////////////////////////////////////

unsigned long ConvertAtoH(char *buffer)
{
   unsigned long output=0;
   int i;

   for (i = 0; i < 8; i++)
   {
      if(buffer[i] <= 'F' && buffer[i] >= 'A')
      {
         output += (buffer[i] - 'A' + 10) << (28 - (i * 4))  ;
      }
      else if(buffer[i] <= 'f' && buffer[i] >= 'a')
      {
         output += (buffer[i] - 'a' + 10) << (28 - (i * 4));
      }
      else if(buffer[i] >= '0' && buffer[i] <= '9')
      {
         output += (buffer[i] - '0') << (28 - (i * 4));
      }
      else
      {
         output=0;
         break;
      }
   }

   return output;
}

//////////////////////////////////////////////////////////////////////////////

int CheckBufferOverRead(unsigned char *buffer)
{
   if (buffer < (work_buffer + f_size))
   {
      return 0;
   }
   else
   {
      return 1;
   }
}

//////////////////////////////////////////////////////////////////////////////

void BufferToBitmap1bpp(unsigned char *buffer, BITMAP *bitmap, unsigned short x, unsigned short y)
{
   int i, i2, i3, j, pix;

   for (i2 = 0; i2 < tile_height; i2++)
   {
      for (i = 0; i < (tile_width / 8); i++)
      {
         if (CheckBufferOverRead(buffer + (i2 * tile_width / 8) + i)) return;

         pix = buffer[(i2 * tile_width / 8) + i];

         for (i3 = 0; i3 < 8; i3++)
         {
            putpixel(bitmap, x + (i * 8) + i3, y + i2,
                     makecol(user_palette[((pix >> 7) & 1)].r * 4,
                             user_palette[((pix >> 7) & 1)].g * 4,
                             user_palette[((pix >> 7) & 1)].b * 4));

            pix <<= 1;
         }
      }
   }

}

//////////////////////////////////////////////////////////////////////////////

void BufferToBitmap2bpp(unsigned char *buffer, BITMAP *bitmap, unsigned short x, unsigned short y)
{
   int i, i2;

   for (i2 = 0; i2 < tile_height; i2++)
   {
      for (i = 0; i < tile_width; i++)
      {
         if (CheckBufferOverRead(buffer + (i2 * tile_width / 4) + (i / 4))) return;

//#ifdef LOW_ENDIAN
//         if ((i % 4) == 0)
//         {
//            dest[i] = src[i / 4] << 6;
//            dest[i] >>= 6;
//         }
//         else if ((i % 4) == 1)
//         {
//            dest[i] = src[i / 4] << 4;
//            dest[i] >>= 6;
//         }
//         else if ((i % 4) == 2)
//         {
//            dest[i] = src[i / 4] << 2;
//            dest[i] >>= 6;
//         }
//         else
//         {
//            dest[i] = src[i / 4] >> 6;
//         }
//#else
         if ((i % 4) == 0)
         {
            putpixel(bitmap,
                     x + i,
                     y + i2,
                     makecol(user_palette[(buffer[(i2 * tile_width / 4) + (i / 4)] & 0xC0) >> 6].r * 4,
                             user_palette[(buffer[(i2 * tile_width / 4) + (i / 4)] & 0xC0) >> 6].g * 4,
                             user_palette[(buffer[(i2 * tile_width / 4) + (i / 4)] & 0xC0) >> 6].b * 4));

         }
         else if ((i % 4) == 1)
         {
            putpixel(bitmap,
                     x + i,
                     y + i2,
                     makecol(user_palette[(buffer[(i2 * tile_width / 4) + (i / 4)] & 0x30) >> 4].r * 4,
                             user_palette[(buffer[(i2 * tile_width / 4) + (i / 4)] & 0x30) >> 4].g * 4,
                             user_palette[(buffer[(i2 * tile_width / 4) + (i / 4)] & 0x30) >> 4].b * 4));

         }
         else if ((i % 4) == 2)
         {
            putpixel(bitmap,
                     x + i,
                     y + i2,
                     makecol(user_palette[(buffer[(i2 * tile_width / 4) + (i / 4)] & 0x0C) >> 2].r * 4,
                             user_palette[(buffer[(i2 * tile_width / 4) + (i / 4)] & 0x0C) >> 2].g * 4,
                             user_palette[(buffer[(i2 * tile_width / 4) + (i / 4)] & 0x0C) >> 2].b * 4));
         }
         else
         {
            putpixel(bitmap,
                     x + i,
                     y + i2,
                     makecol(user_palette[(buffer[(i2 * tile_width / 4) + (i / 4)] & 0x03)].r * 4,
                             user_palette[(buffer[(i2 * tile_width / 4) + (i / 4)] & 0x03)].g * 4,
                             user_palette[(buffer[(i2 * tile_width / 4) + (i / 4)] & 0x03)].b * 4));
         }
//#endif         
      }
   }
}

//////////////////////////////////////////////////////////////////////////////

void BufferToBitmap4bpp(unsigned char *buffer, BITMAP *bitmap, unsigned short x, unsigned short y)
{
   int i, i2;

   acquire_bitmap(bitmap);

   for (i2 = 0; i2 < tile_height; i2++)
   {
      for (i = 0; i < tile_width; i++)
      {
         if (CheckBufferOverRead(buffer + (i2 * tile_width / 2) + (i / 2))) return;
         
         if (i % 2)
         {
            putpixel(bitmap,
                     x + i,
                     y + i2,
                     makecol(user_palette[(buffer[(i2 * tile_width / 2) + (i / 2)] & 0x0F)].r * 4,
                             user_palette[(buffer[(i2 * tile_width / 2) + (i / 2)] & 0x0F)].g * 4,
                             user_palette[(buffer[(i2 * tile_width / 2) + (i / 2)] & 0x0F)].b * 4));
         }
         else
         {
            putpixel(bitmap,
                     x + i,
                     y + i2,
                     makecol(user_palette[(buffer[(i2 * tile_width / 2) + (i / 2)] & 0xF0) / 16].r * 4,
                             user_palette[(buffer[(i2 * tile_width / 2) + (i / 2)] & 0xF0) / 16].g * 4,
                             user_palette[(buffer[(i2 * tile_width / 2) + (i / 2)] & 0xF0) / 16].b * 4));
         }
      }
   }

   release_bitmap(bitmap);
}

//////////////////////////////////////////////////////////////////////////////

void BufferToBitmap8bpp(unsigned char *buffer, BITMAP *bitmap, unsigned short x, unsigned short y)
{
   int i, i2;

   acquire_bitmap(bitmap);


   for (i2 = 0; i2 < tile_height; i2++)
   {
      for (i = 0; i < tile_width; i++)
      {
         if (CheckBufferOverRead(buffer + (i2 * tile_width) + i)) return;

         putpixel(bitmap,
                  x + i,
                  y + i2,
                  makecol(user_palette[buffer[(i2 * tile_width) + i]].r * 4,
                          user_palette[buffer[(i2 * tile_width) + i]].g * 4,
                          user_palette[buffer[(i2 * tile_width) + i]].b * 4));
      }
   }

   release_bitmap(bitmap);
}

//////////////////////////////////////////////////////////////////////////////

void BufferToBitmap16bpp(unsigned char *buffer, BITMAP *bitmap, unsigned short x, unsigned short y)
{
   int i, i2;
   unsigned short pixel=0;
   unsigned char r, g, b;

   for (i2 = 0; i2 < tile_height; i2++)
   {
      for (i = 0; i < tile_width; i++)
      {
         if (CheckBufferOverRead(buffer + (i2 * tile_width * 2) + (i * 2) + 1)) return;

         pixel = (buffer[(i2 * tile_width * 2) + (i * 2)] << 8) +
                  buffer[(i2 * tile_width * 2) + (i * 2) + 1];

         if (pixel & 0x8000)
         {
            pixel &= ~0x8000;
         }
            r = pixel << 3;
            g = pixel >> 5;
            g <<= 3;
            b = pixel >> 10;
            b <<= 3;

            r >>= 3;
            g >>= 3;
            b >>= 3;

            putpixel(bitmap, x + i, y + i2, makecol(r << 3, g << 3, b << 3));
//         }
      }
   }
}

//////////////////////////////////////////////////////////////////////////////

void BufferToBitmap24bpp(unsigned char *buffer, BITMAP *bitmap, unsigned short x, unsigned short y)
{
   int i, i2;

   acquire_bitmap(bitmap);

   for (i2 = 0; i2 < tile_height; i2++)
   {
      for (i = 0; i < tile_width; i++)
      {
         if (CheckBufferOverRead(buffer + (i2 * tile_width * 4) + (i * 4) + 3)) return;

         putpixel(bitmap,
                  x + i,
                  y + i2,
                  makecol(buffer[(i2 * tile_width * 4) + (i * 4) + 3],
                          buffer[(i2 * tile_width * 4) + (i * 4) + 2],
                          buffer[(i2 * tile_width * 4) + (i * 4) + 1]));
      }
   }

   release_bitmap(bitmap);
}

//////////////////////////////////////////////////////////////////////////////

void DrawPage (HWND hWnd)
{
   unsigned long temp_long;
   unsigned short x=0,y=0;
   int i, i2;
   int done=0;
   char temp_string[120];

   clear_bitmap(double_buffer);


   num_row = (MAX_HEIGHT / tile_height);

   if (auto_col)
      num_col = (MAX_WIDTH / tile_width);
   else
   {
      if (usr_col <= (int)(MAX_WIDTH / tile_width))
         num_col = usr_col;
      else
         num_col = (MAX_WIDTH / tile_width);
   }
   if (auto_col)
      sprintf(temp_string, "%s - Offset: 0x%08x \\ t. width = %d \\ t. height = %d \\ bpp = %d \\ col num = %d \\ Auto col = on", szAppName, f_offset, tile_width, tile_height, bpp, num_col);
   else
      sprintf(temp_string, "%s - Offset: 0x%08x \\ t. width = %d \\ t. height = %d \\ bpp = %d \\ col num = %d \\ Auto col = off", szAppName, f_offset, tile_width, tile_height, bpp, num_col);

   SetWindowText(hWnd, temp_string);


   for (i2 = 0; i2 < num_row; i2++)
   {
      for (i = 0; i < num_col; i++)
      {
         switch (bpp)
         {
            case 0:
            {
               // Custom
               break;
            }
            case 1:
            {
               BufferToBitmap1bpp(work_buffer + f_offset + (((tile_width * tile_height / 8) * num_col) * i2) + (i * (tile_width * tile_height / 8)), double_buffer, i * tile_width, i2 * tile_height);

               break;
            }
            case 2:
            {
               BufferToBitmap2bpp(work_buffer + f_offset + (((tile_width * tile_height / 4) * num_col) * i2) + (i * (tile_width * tile_height / 4)), double_buffer, i * tile_width, i2 * tile_height);

               break;
            }
            case 4:
            {
               BufferToBitmap4bpp(work_buffer + f_offset + (((tile_width * tile_height / 2) * num_col) * i2) + (i * (tile_width * tile_height / 2)), double_buffer, i * tile_width, i2 * tile_height);

               break;
            }
            case 8:
            {
               BufferToBitmap8bpp(work_buffer + f_offset + (((tile_width * tile_height) * num_col) * i2) + (i * (tile_width * tile_height)), double_buffer, i * tile_width, i2 * tile_height);

               break;
            }
            case 16:
            {
               BufferToBitmap16bpp(work_buffer + f_offset + (((tile_width * tile_height * 2) * num_col) * i2) + (i * (tile_width * tile_height * 2)), double_buffer, i * tile_width, i2 * tile_height);

               break;
            }
            case 24:
            {
               BufferToBitmap24bpp(work_buffer + f_offset + (((tile_width * tile_height * 4) * num_col) * i2) + (i * (tile_width * tile_height * 4)), double_buffer, i * tile_width, i2 * tile_height);

               break;
            }
            default: break;
         }
      }
   }
}

//////////////////////////////////////////////////////////////////////////////

void UpdateScreen()
{
   show_mouse(NULL);
   acquire_bitmap(screen);
   stretch_blit(double_buffer, screen, 0, 0, MAX_WIDTH, MAX_HEIGHT, 0, 0, 640, 480);
   release_bitmap(screen);
   show_mouse(screen);
}
